<?php

namespace App\Http\Controllers;

use Illuminate\Routing\Controller;
use App\Models\Customer;
use App\Services\CustomerPasswordResetService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Tymon\JWTAuth\Facades\JWTAuth;

class CustomerAuthController extends Controller
{
    protected $passwordResetService;
    /**
     * Create a new AuthController instance.
     *
     * @return void
     */
    public function __construct(CustomerPasswordResetService $passwordResetService)
    {
        $this->middleware('auth:customer', ['except' => ['login', 'register', 'forgotPassword', 'resetPassword', 'verifyResetToken']]);
        $this->passwordResetService = $passwordResetService;
    }

    /**
     * Register a new customer.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function register(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'firstname' => 'required|string|max:255',
            'lastname' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:customers',
            'password' => 'required|string|min:6|confirmed',
            'phone' => 'nullable|string|max:20',
            'country' => 'nullable|string|max:100',
            'address' => 'nullable|string|max:500',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => implode(" | ", $validator->errors()->all()),
                'errors' => $validator->errors()
            ], 422);
        }

        $customer = Customer::create([
            'firstname' => $request->firstname,
            'lastname' => $request->lastname,
            'email' => $request->email,
            'password' => $request->password, // Will be hashed automatically
            'phone' => $request->phone,
            'country' => $request->country,
            'address' => $request->address,
            'status' => 'active'
        ]);

        $token = Auth::guard('customer')->login($customer);

        return response()->json([
            'success' => true,
            'message' => 'Customer registered successfully',
            'customer' => $customer,
            'access_token' => $token,
            'token_type' => 'bearer',
            'expires_in' => Auth::guard('customer')->factory()->getTTL() * 60
        ], 201);
    }

    /**
     * Get a JWT via given credentials.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
            'password' => 'required|string|min:6',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $credentials = $request->only('email', 'password');

        if (!$token = Auth::guard('customer')->attempt($credentials)) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid credentials'
            ], 401);
        }

        $customer = Auth::guard('customer')->user();
        $customer = Customer::where("id", $customer->id)->first();
        $customer->wallet;

        if (!$customer->isActive()) {
            Auth::guard('customer')->logout();
            return response()->json([
                'success' => false,
                'message' => 'Account is not active'
            ], 403);
        }

        return $this->respondWithToken($token);
    }

    /**
     * Get the authenticated Customer.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function profile()
    {
        $id = Auth::guard('customer')->user()->id;
        $customer = Customer::where("id", $id)->first();
        $wallet = $customer->wallet->balance;
        $customer->balance = $wallet;
        return response()->json([
            'success' => true,
            'customer' => $customer
        ]);
    }

    /**
     * Update the authenticated customer's profile.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateProfile(Request $request)
    {
        $customer = Auth::guard('customer')->user();

        $validator = Validator::make($request->all(), [
            'firstname' => 'sometimes|required|string|max:255',
            'lastname' => 'sometimes|required|string|max:255',
            'email' => 'sometimes|required|string|email|max:255|unique:customers,email,' . $customer->id,
            'phone' => 'nullable|string|max:20',
            'country' => 'nullable|string|max:100',
            'address' => 'nullable|string|max:500',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => implode(" | ", $validator->errors()->all()),
                'errors' => $validator->errors()
            ], 422);
        }

        $customer->update($request->only([
            'firstname',
            'lastname',
            'email',
            'phone',
            'country',
            'address'
        ]));

        return response()->json([
            'success' => true,
            'message' => 'Profile updated successfully',
            'customer' => $customer->fresh()
        ]);
    }

    /**
     * Change password.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function changePassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'current_password' => 'required|string',
            'new_password' => 'required|string|min:6|confirmed',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $customer = Auth::guard('customer')->user();

        if (!Hash::check($request->current_password, $customer->password)) {
            return response()->json([
                'success' => false,
                'message' => 'Current password is incorrect'
            ], 422);
        }

        $customer->update([
            'password' => $request->new_password
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Password changed successfully'
        ]);
    }

    /**
     * Log the customer out (Invalidate the token).
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function logout()
    {
        Auth::guard('customer')->logout();

        return response()->json([
            'success' => true,
            'message' => 'Successfully logged out'
        ]);
    }

    /**
     * Refresh a token.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function refresh()
    {
        return $this->respondWithToken(Auth::guard('customer')->refresh());
    }

    /**
     * Get the token array structure.
     *
     * @param  string $token
     * @return \Illuminate\Http\JsonResponse
     */
    protected function respondWithToken($token)
    {
        return response()->json([
            'success' => true,
            'customer' => Auth::guard('customer')->user(),
            'access_token' => $token,
            'token_type' => 'bearer',
            'expires_in' => Auth::guard('customer')->factory()->getTTL() * 60
        ]);
    }



    /**
     * Send password reset link.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function forgotPassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $result = $this->passwordResetService->sendResetLink($request->email);

        return response()->json($result, $result['success'] ? 200 : 422);
    }

    /**
     * Reset customer password.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function resetPassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
            'token' => 'required|string',
            'password' => 'required|string|min:6|confirmed',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $result = $this->passwordResetService->resetPassword(
            $request->email,
            $request->token,
            $request->password
        );

        return response()->json($result, $result['success'] ? 200 : 422);
    }

    /**
     * Verify password reset token.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function verifyResetToken(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
            'token' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $isValid = $this->passwordResetService->verifyToken(
            $request->email,
            $request->token
        );

        return response()->json([
            'success' => $isValid,
            'message' => $isValid ? 'Token is valid' : 'Invalid or expired token'
        ]);
    }
}
