<?php


namespace App\Http\Controllers;

use App\Models\FundingTransaction;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;
use App\Models\Customer;
use App\Notifications\FundingConfirmedNotification;

class FundingTransactionController extends Controller
{
    /**
     * Display a listing of funding transactions.
     */
    public function index(Request $request): JsonResponse
    {
        $query = FundingTransaction::with('customer');

        // Filter by customer if provided
        if ($request->has('customer_id')) {
            $query->forCustomer($request->customer_id);
        }

        // Filter by status if provided
        if ($request->has('status')) {
            $query->byStatus($request->status);
        }

        // Filter by verification status if provided
        if ($request->has('verified')) {
            if ($request->verified === 'verified') {
                $query->verified();
            } elseif ($request->verified === 'unverified') {
                $query->unverified();
            }
        }

        // Pagination
        $perPage = $request->get('per_page', 15);
        $transactions = $query->orderBy('created_at', 'desc')->paginate($perPage);

        return response()->json([
            'success' => true,
            'data' => $transactions
        ]);
    }

    /**
     * Store a newly created funding transaction.
     */
    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|exists:customers,id',
            'payment_method' => 'required|string|max:255',
            'address' => 'required|string',
            'amount' => 'required|numeric|min:0.01',
            'image_url' => 'nullable|string|url',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $transaction = FundingTransaction::create($request->all());

        return response()->json([
            'success' => true,
            'message' => 'Funding transaction created successfully',
            'data' => $transaction->load('customer')
        ], 201);
    }

    /**
     * Display the specified funding transaction.
     */
    public function show(FundingTransaction $fundingTransaction): JsonResponse
    {
        return response()->json([
            'success' => true,
            'data' => $fundingTransaction->load('customer')
        ]);
    }

    /**
     * Update the specified funding transaction.
     */
    public function update(Request $request, FundingTransaction $fundingTransaction): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'customer_id' => 'sometimes|exists:users,id',
            'payment_method' => 'sometimes|string|max:255',
            'address' => 'sometimes|string',
            'status' => 'sometimes|in:pending,processing,completed,failed,cancelled',
            'verified' => 'sometimes|in:verified,unverified',
            'amount' => 'sometimes|numeric|min:0.01',
            'image_url' => 'nullable|string|url',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $fundingTransaction->update($request->all());

        return response()->json([
            'success' => true,
            'message' => 'Funding transaction updated successfully',
            'data' => $fundingTransaction->load('customer')
        ]);
    }

    /**
     * Remove the specified funding transaction.
     */
    public function destroy(FundingTransaction $fundingTransaction): JsonResponse
    {
        $fundingTransaction->delete();

        return response()->json([
            'success' => true,
            'message' => 'Funding transaction deleted successfully'
        ]);
    }

    /**
     * Verify a funding transaction.
     */
    public function proveImageUrl(FundingTransaction $fundingTransaction, Request $request): JsonResponse
    {
        $fundingTransaction->update(['image_url' => $request->image_url]);

        return response()->json([
            'success' => true,
            'message' => 'Prove uploaded successfully',
            'data' => $fundingTransaction
        ]);
    }


    public function confirmFunding(FundingTransaction $fundingTransaction)
    {
        $customer = Customer::where("id", $fundingTransaction->customer_id)->first();

        try {
            $customer->deposit($fundingTransaction->amount);
            $fundingTransaction->update([
                "status" => FundingTransaction::STATUS_COMPLETED
            ]);
            $customer->notify(new FundingConfirmedNotification($fundingTransaction, $customer));
        } catch (\Throwable $th) {
            // Log the error for debugging
            logs()->error('Deposit failed for customer ' . $customer->id . ': ' . $th->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to process deposit',
                'error' => $th->getMessage()
            ], 500);
        }


        return response()->json([
            'success' => true,
            'message' => 'Funding confirmed successfully',
            'data' => $fundingTransaction,
            'customer' => $customer
        ]);
    }


    /**
     * Verify a funding transaction.
     */
    public function verify(FundingTransaction $fundingTransaction): JsonResponse
    {
        $fundingTransaction->update(['verified' => FundingTransaction::VERIFIED]);

        return response()->json([
            'success' => true,
            'message' => 'Transaction verified successfully',
            'data' => $fundingTransaction->load('customer')
        ]);
    }

    /**
     * Update transaction status.
     */
    public function updateStatus(Request $request, FundingTransaction $fundingTransaction): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'status' => 'required|in:pending,processing,completed,failed,cancelled'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $fundingTransaction->update(['status' => $request->status]);

        return response()->json([
            'success' => true,
            'message' => 'Transaction status updated successfully',
            'data' => $fundingTransaction->load('customer')
        ]);
    }

    /**
     * Get transactions by customer.
     */
    public function getByCustomer(Request $request, int $customerId): JsonResponse
    {
        $perPage = $request->get('per_page', 15);
        $transactions = FundingTransaction::forCustomer($customerId)
            ->with('customer')
            ->orderBy('created_at', 'desc')
            ->paginate($perPage);

        return response()->json([
            'success' => true,
            'data' => $transactions
        ]);
    }
}
