<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use App\Models\Investment;

class CustomerInvestment extends Model
{
    use HasFactory;

    protected $table = 'customer_investments';
    protected $fillable = [
        'customer_id',
        'investment_id',
        'amount',
        'start_seconds',
        'end_seconds',
        'status',
    ];
    protected $appends = ['investment', 'ested_amount', 'percentage_progress', 'start_date', 'remaining_days', 'present_earnings', 'percentage_completed', 'end_date'];

    public function getInvestmentAttribute()
    {
        $investment = Investment::find($this->investment_id);
        return $investment ? $investment : null;
    }

    public function getEstedAmountAttribute()
    {
        $investment = Investment::find($this->investment_id);
        if ($investment) {
            $percentage = floatval($investment->percentage);
            $bonus = floatval($investment->bonus);
            $investedAmount = floatval($this->amount);

            // Calculate the estimated amount
            $estimatedAmount = $investedAmount + ($investedAmount * $percentage / 100) + $bonus;
            return round($estimatedAmount, 2);
        }
        return 0;
    }

    public function getPercentageProgressAttribute()
    {
        $investment = Investment::find($this->investment_id);
        if ($investment) {
            $currentTime = time();
            $startTime = intval($this->start_seconds);
            $endTime = intval($this->end_seconds);

            if ($currentTime < $startTime) {
                return 0;
            } elseif ($currentTime >= $endTime) {
                return 100;
            } else {
                $totalDuration = $endTime - $startTime;
                $elapsedDuration = $currentTime - $startTime;
                $percentageProgress = ($elapsedDuration / $totalDuration) * 100;
                return round($percentageProgress, 2);
            }
        }
        return 0;
    }

    public function getStartDateAttribute()
    {
        return date('Y-m-d H:i:s', $this->start_seconds);
    }

    public function getRemainingDaysAttribute()
    {
        $currentTime = time();
        $endTime = intval($this->end_seconds);

        if ($currentTime >= $endTime) {
            return 0;
        } else {
            $remainingSeconds = $endTime - $currentTime;
            return ceil($remainingSeconds / 86400); // 86400 seconds in a day
        }
    }

    public function getPresentEarningsAttribute()
    {
        $investment = Investment::find($this->investment_id);
        if ($investment) {
            $percentage = floatval($investment->percentage);
            $investedAmount = floatval($this->amount);

            $currentTime = time();
            $startTime = intval($this->start_seconds);
            $endTime = intval($this->end_seconds);

            if ($currentTime < $startTime) {
                return 0;
            } elseif ($currentTime >= $endTime) {
                // Full earnings
                $earnings = ($investedAmount * $percentage / 100);
                return round($earnings, 2);
            } else {
                // Partial earnings based on elapsed time
                $totalDuration = $endTime - $startTime;
                $elapsedDuration = $currentTime - $startTime;
                $partialEarnings = ($investedAmount * $percentage / 100) * ($elapsedDuration / $totalDuration);
                return round($partialEarnings, 2);
            }
        }
        return 0;
    }

    public function getPercentageCompletedAttribute()
    {
        $investment = Investment::find($this->investment_id);
        if ($investment) {
            $currentTime = time();
            $startTime = intval($this->start_seconds);
            $endTime = intval($this->end_seconds);

            if ($currentTime < $startTime) {
                return 0;
            } elseif ($currentTime >= $endTime) {
                return floatval($investment->percentage);
            } else {
                $totalDuration = $endTime - $startTime;
                $elapsedDuration = $currentTime - $startTime;
                $percentageCompleted = (floatval($investment->percentage)) * ($elapsedDuration / $totalDuration);
                return round($percentageCompleted, 2);
            }
        }
        return 0;
    }
    public function getEndDateAttribute()
    {
        return date('Y-m-d H:i:s', $this->end_seconds);
    }
}
