<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class FundingTransaction extends Model
{
    use HasFactory;

    protected $fillable = [
        'customer_id',
        'payment_method',
        'address',
        'status',
        'verified',
        'image_url',
        'amount',
    ];

    protected $casts = [
        'amount' => 'decimal:2',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    protected $with = [
        "customer"
    ];

    protected $appends = ['full_name'];

    // Constants for status values
    public const STATUS_PENDING = 'pending';
    public const STATUS_PROCESSING = 'processing';
    public const STATUS_COMPLETED = 'completed';
    public const STATUS_FAILED = 'failed';
    public const STATUS_CANCELLED = 'cancelled';

    // Constants for verification status
    public const VERIFIED = 'verified';
    public const UNVERIFIED = 'unverified';

    /**
     * Get the customer that owns the funding transaction.
     */
    public function customer(): BelongsTo
    {
        return $this->belongsTo(User::class, 'customer_id');
    }

    /**
     * Scope a query to only include verified transactions.
     */
    public function scopeVerified($query)
    {
        return $query->where('verified', self::VERIFIED);
    }

    /**
     * Scope a query to only include unverified transactions.
     */
    public function scopeUnverified($query)
    {
        return $query->where('verified', self::UNVERIFIED);
    }

    /**
     * Scope a query to filter by status.
     */
    public function scopeByStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Scope a query to only include transactions for a specific customer.
     */
    public function scopeForCustomer($query, $customerId)
    {
        return $query->where('customer_id', $customerId);
    }

    /**
     * Get formatted amount with currency symbol.
     */
    public function getFormattedAmountAttribute(): string
    {
        return '$' . number_format($this->amount, 2);
    }

    /**
     * Check if transaction is verified.
     */
    public function isVerified(): bool
    {
        return $this->verified === self::VERIFIED;
    }

    /**
     * Check if transaction is completed.
     */
    public function isCompleted(): bool
    {
        return $this->status === self::STATUS_COMPLETED;
    }

    public function getFullNameAttribute()
    {
        $customer = Customer::where("id", $this->customer_id)->first();

        return ucfirst($customer->firstname) . ' ' . ucfirst($customer->lastname);
    }
}
