<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Investment extends Model
{
    use HasFactory, SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'min_amount',
        'max_amount',
        'investment_name',
        'investment_description',
        'bonus',
        'percentage',
        'duration',
        'status',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'min_amount' => 'decimal:2',
        'max_amount' => 'decimal:2',
        'bonus' => 'decimal:2',
        'duration' => 'integer',
        'status' => 'string',
    ];

    /**
     * Get the route key for the model.
     */
    public function getRouteKeyName(): string
    {
        return 'id';
    }

    /**
     * Scope a query to only include active investments.
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    /**
     * Scope a query to only include inactive investments.
     */
    public function scopeInactive($query)
    {
        return $query->where('status', 'inactive');
    }

    /**
     * Check if the investment is active.
     */
    public function isActive(): bool
    {
        return $this->status === 'active';
    }

    /**
     * Check if the investment amount is within range.
     */
    public function isAmountInRange(float $amount): bool
    {
        return $amount >= $this->min_amount && $amount <= $this->max_amount;
    }

    /**
     * Get formatted min amount.
     */
    public function getFormattedMinAmountAttribute(): string
    {
        return number_format($this->min_amount, 2);
    }

    /**
     * Get formatted max amount.
     */
    public function getFormattedMaxAmountAttribute(): string
    {
        return number_format($this->max_amount, 2);
    }

    /**
     * Get formatted bonus.
     */
    public function getFormattedBonusAttribute(): string
    {
        return number_format($this->bonus, 2);
    }

    /**
     * Get duration in human readable format.
     */
    public function getFormattedDurationAttribute(): string
    {
        if ($this->duration < 30) {
            return $this->duration . ' days';
        } elseif ($this->duration < 365) {
            $months = round($this->duration / 30, 1);
            return $months . ' months';
        } else {
            $years = round($this->duration / 365, 1);
            return $years . ' years';
        }
    }

    /**
     * Get duration in months.
     */
    public function getDurationInMonths(): float
    {
        return round($this->duration / 30, 2);
    }

    /**
     * Get duration in years.
     */
    public function getDurationInYears(): float
    {
        return round($this->duration / 365, 2);
    }
}
