<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class PaymentType extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'value',
        'type',
        'admin_id',
        'customer_id',
        'isActive',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'isActive' => 'boolean',
    ];

    /**
     * Get the admin that owns the payment type.
     */
    public function admin()
    {
        return $this->belongsTo(Admin::class, 'admin_id');
    }

    /**
     * Get the customer that owns the payment type.
     */
    public function customer()
    {
        return $this->belongsTo(Customer::class, 'customer_id');
    }

    /**
     * Scope a query to only include active payment types.
     */
    public function scopeActive($query)
    {
        return $query->where('isActive', 1);
    }

    /**
     * Scope a query to only include inactive payment types.
     */
    public function scopeInactive($query)
    {
        return $query->where('isActive', 0);
    }

    /**
     * Scope a query to filter by type.
     */
    public function scopeByType($query, $type)
    {
        return $query->where('type', $type);
    }

    /**
     * Check if payment type is active.
     */
    public function isActive()
    {
        return $this->isActive === 1;
    }

    /**
     * Activate the payment type.
     */
    public function activate()
    {
        return $this->update(['isActive' => 1]);
    }

    /**
     * Deactivate the payment type.
     */
    public function deactivate()
    {
        return $this->update(['isActive' => 0]);
    }
}
