<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use App\Models\FundingTransaction;
use App\Models\Customer;

class FundingConfirmedNotification extends Notification
{
    protected $fundingTransaction;
    protected $customer;

    /**
     * Create a new notification instance.
     */
    public function __construct(FundingTransaction $fundingTransaction, Customer $customer)
    {
        $this->fundingTransaction = $fundingTransaction;
        $this->customer = $customer;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('Funding Confirmed - Transaction Successful')
            ->greeting('Hello ' . $this->customer->firstname . '!')
            ->line('Your funding transaction has been successfully confirmed and processed.')
            ->line('Transaction Details:')
            ->line('Amount: $' . number_format($this->fundingTransaction->amount, 2))
            ->line('Transaction ID: ' . md5($this->fundingTransaction->id))
            ->line('Status: ' . ucfirst(strtolower($this->fundingTransaction->status)))
            ->line('Date: ' . $this->fundingTransaction->updated_at->format('M d, Y H:i'))
            ->line('Your account balance has been updated accordingly.')
            ->line('Thank you for using our service!');
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'title' => 'Funding Confirmed',
            'message' => 'Your funding of $' . number_format($this->fundingTransaction->amount, 2) . ' has been successfully processed.',
            'transaction_id' => $this->fundingTransaction->id,
            'amount' => $this->fundingTransaction->amount,
            'status' => $this->fundingTransaction->status,
            'confirmed_at' => $this->fundingTransaction->updated_at,
            'type' => 'funding_confirmed'
        ];
    }

    /**
     * Get the notification's database type.
     */
    public function databaseType(object $notifiable): string
    {
        return 'funding-confirmed';
    }
}
