<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Messages\DatabaseMessage;
use Illuminate\Notifications\Notification;
use App\Models\CustomerInvestment;

class InvestmentJoinedNotification extends Notification
{

    protected $customerInvestment;
    protected $investment;

    public function __construct(CustomerInvestment $customerInvestment)
    {
        $this->customerInvestment = $customerInvestment;
        $this->investment = $customerInvestment->investment;
    }

    public function via($notifiable)
    {
        return ['database', 'mail']; // Add 'sms' if you have SMS setup
    }

    public function toMail($notifiable)
    {
        return (new MailMessage)
            ->subject('Investment Joined Successfully')
            ->greeting('Hello ' . $notifiable->name . '!')
            ->line('You have successfully joined the investment: ' . $this->investment->investment_name)
            ->line('Investment Amount: $' . number_format($this->customerInvestment->amount, 2))
            ->line('Duration: ' . $this->investment->duration . ' days')
            ->line('Expected Return: ' . $this->investment->percentage . '%')
            ->line('Expected Return Amount: $' . number_format($this->customerInvestment->amount * ($this->investment->percentage / 100)))
            ->line('Start Date: ' . now()->format('Y-m-d H:i:s'))
            ->line('End Date: ' . now()->addDays($this->investment->duration)->format('Y-m-d H:i:s'))
            ->line('Thank you for investing with us!');
    }

    public function toDatabase($notifiable)
    {
        return [
            'title' => 'Investment Joined Successfully',
            'message' => 'You have successfully joined the investment: ' . $this->investment->investment_name,
            'investment_id' => $this->investment->id,
            'customer_investment_id' => $this->customerInvestment->id,
            'amount' => $this->customerInvestment->amount,
            'investment_name' => $this->investment->investment_name,
            'duration' => $this->investment->duration,
            'return_rate' => $this->investment->percentage ?? 'N/A',
            'start_date' => now()->format('Y-m-d H:i:s'),
            'end_date' => now()->addDays($this->investment->duration)->format('Y-m-d H:i:s'),
            'type' => 'investment_joined'
        ];
    }

    public function toArray($notifiable)
    {
        return $this->toDatabase($notifiable);
    }
}
